/*******************************************************************************
********************************************************************************
** COPYRIGHT:    (c) 2013 Rohde & Schwarz
** MODULE:       CSCPIHelper_PR100.h
** LANGUAGE:     C/C++
** AUTHOR:       
** ABSTRACT:     Helper class to encapsulate SCPI access for PR100
** PREMISES:
** REMARKS:
** HISTORY:
********************************************************************************/
#ifndef CSCPI_HELPER_H
#define CSCPI_HELPER_H


/* INCLUDE FILES ***************************************************************/
#include <string>
#include <functional>

#ifndef _WINSOCKAPI_
#include <winsock2.h> //For SOCKET
#endif

/* CLASSES ***************************************************************/
class CSCPIHelper
{
public:
    enum BLOCKDATA_STATUS_T
    {
        BLOCKDATA_STATUS_DONE, //all data in the block data have been received.
        BLOCKDATA_STATUS_FAIL, //error receiving data.
        BLOCKDATA_STATUS_DATA  //data received.
    };

    struct BLOCKDATA_T
    {
        //Current Transfer status
        BLOCKDATA_STATUS_T status;        //status of this blockdata
        int                totalDataSize; //size of the entire block data

        //Data packet information, only valid if status == BLOCKDATA_STATUS_DATA
        void * pBufferData;
        int    bufferSize;
        int    offset;      //start offset of this data fragment within the entire block data
    };

    //Callback function typedef
    typedef std::function<void (void*, const BLOCKDATA_T&)> CallbackFunc;

    enum OPTIONS_T
    {
        OPTION_PS = 1 << 0,
        OPTION_IR = 1 << 1,
        OPTION_RC = 1 << 2,
        OPTION_ET = 1 << 3,
        OPTION_FS = 1 << 4,
        OPTION_FP = 1 << 5,
        OPTION_FE = 1 << 6,
        OPTION_GP = 1 << 7,
        OPTION_DF = 1 << 8
    };


public:
    //Constructor
    CSCPIHelper(std::string remoteIP, unsigned short port);

    //Connect/Disconnect to/from the client
    bool  Connect();
    bool  Disconnect();

    //Send a SCPI command
    //
    //Returns
    //  0 = no error
    // -1 = error writing to device
    int   SendCmd( const  char *pBuffer );
    int   SendCmd( const  std::string str ) { return SendCmd(str.c_str()); }
    
    //Receive ascii data from SCPI device, with timeout.
    //
    //Parameters:
    // pBuffer   = pointer to buffer to receive data
    // length    = length of pBuffer. On return, contains the length of data written into pBuffer
    // msTimeOut = timeout in milliseconds
    // 
    //Returns 
    //  0 = No error
    // -1 = timeout
    // -2 = socket error
    // -3 = null pointer parameter or buffer length 0
    // -4 = buffer full. Caller should call this function again to get the rest of the response.
    //
    //Note:
    // - Parser stops at the first <CR><LF> characters encountered (signifies the end of one response),
	//   or if the buffer is full.
    // - The <CR><LF> characters are discarded.
    // - Do not use with Block Data responses.
    // 
    int GetResponseWithTimeout(char *pBuffer, int& length, unsigned int msTimeOut);
    
    //Receive binary data (used for block data), with timeout
    //
    //Parameters:
    // pBuffer   = pointer to buffer to receive data
    // length    = length of pBuffer. On return, contains the length of data written into pBuffer
    // msTimeOut = timeout in milliseconds
    // 
    //Returns 
    //  0 = No error
    // -1 = timeout
    // -2 = socket error
    // -3 = null pointer parameter or buffer length 0
    //
    int GetBinaryResponseWithTimeout(char *pBuffer, int& length, unsigned int msTimeOut);

    //Gets BlockData, parses the block data from the response
    // and delivers the data to caller via the callback function.
	// The caller opaque pointer "pPrivate" will be passed back to the caller in the callback.
    //Returns
    //  0 = no error
    // -1 = timeout
    // -2 = corrupted block data header
    int GetBlockData(CallbackFunc callback, void * pPrivate);

    //Gets and parses the SCPI command response from "syst:err:code:all?" for errors.
    //  0 = No error
    // -1 = timeout getting response from device
    // -2 = socket error
    // otherwise = SCPI error
    int   GetLastCmdErr(); 

    //Get the properties of the device
    std::string DeviceName   ( void ) { return m_sDeviceName; }
    std::string DeviceAddress( void ) { return m_sIP; }
    int         DevicePort   ( void ) { return m_nPort; }
    int         DeviceSN     ( void ) { return m_nSN; }
    std::string DeviceFWVer  ( void ) { return m_sFWVer; }
    bool        GetDeviceOptions ( unsigned int& options ) ;

    //Static helper functions
    static char * Tokenise(char *str, const char *delim, char **nextp);
    static char * TrimBefore(char *str);

    //Debug functions
    void  SetDebugLevel(bool bEnable) {m_bDebugOutput = bEnable;}

protected:
    static const int READ_SIZE_BYTES           = 256;
    static const int READ_BLOCKDATA_SIZE_BYTES = 1500; //controls the buffer size used on the stack for block data transfer.
    static const int READ_TIMEOUT_MS           = 2000;

    /* retrieve R&S device information */
    bool        GetDeviceIdentify( void );
    bool        ResolveIP();
  
    std::string         m_sIP;
    unsigned long       m_nIP;
    int                 m_nPort;
    SOCKET              m_Socket;

    unsigned int        m_nOptions;
    std::string         m_sDeviceName;
    int                 m_nSN;
    std::string         m_sFWVer;

    bool                m_bDebugOutput;
};

#endif